<?php
/**
 * Varre um diretrio e processa arquivos JSON com padro:
 *   JsonOVDeviceToDBServer-[A-Za-z0-9]+-[0-9]+.json
 *
 * Compatvel com PHP 7.x
 */

$dir   = __DIR__ . '/sinais/AProcessar';
$regex = '/^JsonOVDeviceToDBServer-[A-Za-z0-9]+-\d+\.json$/';
$prefix = 'JsonOVDeviceToDBServer';

/* -------------------- Helpers -------------------- */

function bool_to_str($v) {
    return is_bool($v) ? ($v ? 'true' : 'false') : $v;
}

function print_kv($label, $value, $indent = 0) {
    $pad = str_repeat(' ', $indent);
    if (is_bool($value)) $value = bool_to_str($value);
    if (is_array($value)) {
        echo "{$pad}{$label}: " . json_encode($value, JSON_UNESCAPED_UNICODE) . PHP_EOL;
    } else {
        echo "{$pad}{$label}: {$value}" . PHP_EOL;
    }
}

function print_section($title, $indent = 0) {
    $pad = str_repeat(' ', $indent);
    echo PHP_EOL . "{$pad}== {$title} ==" . PHP_EOL;
}

function decode_json_strict($raw) {
    // Se o arquivo vier com encoding problemtico, normalize para UTF-8:
    if (!mb_check_encoding($raw, 'UTF-8')) {
        $raw = mb_convert_encoding($raw, 'UTF-8', 'UTF-8, ISO-8859-1, Windows-1252');
    }
    $data = json_decode($raw, true);
    if (json_last_error() !== JSON_ERROR_NONE) {
        throw new RuntimeException('JSON invlido: ' . json_last_error_msg());
    }
    return $data;
}

/**
 * Imprime todas as chaves de um array (recursivo simples para arrays de nvel 1 ou 2).
 * til como .fallback. para garantir que nada passe batido.
 */
function print_all_keys_generic(array $arr, $title = 'Outros campos', $indent = 2) {
    if (empty($arr)) return;
    print_section($title, $indent - 2);
    foreach ($arr as $k => $v) {
        if (is_array($v)) {
            print_kv($k, json_encode($v, JSON_UNESCAPED_UNICODE), $indent);
        } else {
            print_kv($k, $v, $indent);
        }
    }
}

/* -------------------- Impresso detalhada do pedido -------------------- */

function print_pedido(array $pedido, $indentBase = 0) {
    $ind1 = $indentBase + 2;
    $ind2 = $indentBase + 4;

    // Sees conhecidas (todas as propriedades sero impressas via foreach)
    $sections = [
        'Emi', 'Dest', 'TOper', 'EndeE', 'EndeD', 'EDTOp'
    ];

    foreach ($sections as $sec) {
        if (isset($pedido[$sec]) && is_array($pedido[$sec])) {
            print_section($sec, $indentBase);
            foreach ($pedido[$sec] as $k => $v) {
                print_kv($k, $v, $ind1);
            }
        }
    }

    // Itens (imprime todas as propriedades de cada item)
    if (!empty($pedido['aItens']) && is_array($pedido['aItens'])) {
        print_section('aItens', $indentBase);
        foreach ($pedido['aItens'] as $idx => $item) {
            echo str_repeat(' ', $ind1) . "- Item #".($idx+1) . PHP_EOL;
            if (is_array($item)) {
                foreach ($item as $k => $v) {
                    print_kv($k, $v, $ind2);
                }
            } else {
                print_kv('valor', $item, $ind2);
            }
        }
    }

    // Campos simples e outras sees conhecidas
    $simples = [
        'tEntrega', 'sObs'
    ];
    foreach ($simples as $k) {
        if (array_key_exists($k, $pedido)) {
            print_kv($k, $pedido[$k], $indentBase);
        }
    }

    if (isset($pedido['CPgmto']) && is_array($pedido['CPgmto'])) {
        print_section('CPgmto', $indentBase);
        foreach ($pedido['CPgmto'] as $k => $v) {
            print_kv($k, $v, $ind1);
        }
    }

    // Totais (imprime todos os que existirem)
    $totaisKeys = [
        'nTot_vIPI','nTot_vPIS','nTot_vCOFINS','nTot_vICMS','nTot_vICMSDeson',
        'nTot_vBCST','nTot_vST','nVFCPUFDest','nVICMSUFDest','nVICMSUFRemet',
        'nICMSTot_vFCP','nICMSTot_vFCPST','nICMSTot_vFCPSTRet'
    ];
    $temTotais = false;
    foreach ($totaisKeys as $k) {
        if (array_key_exists($k, $pedido)) { $temTotais = true; break; }
    }
    if ($temTotais) {
        print_section('Totais', $indentBase);
        foreach ($totaisKeys as $k) {
            if (array_key_exists($k, $pedido)) {
                print_kv($k, $pedido[$k], $ind1);
            }
        }
    }

    // Campos finais
    foreach (['jItem','pId','tCriacao'] as $k) {
        if (array_key_exists($k, $pedido)) {
            print_kv($k, $pedido[$k], $indentBase);
        }
    }

    // Fallback: se houver chaves no cobertas acima, imprime todas
    $chavesCobertas = array_flip(array_merge(
        $sections,
        ['aItens','tEntrega','sObs','CPgmto'],
        $totaisKeys,
        ['jItem','pId','tCriacao']
    ));
    $resto = [];
    foreach ($pedido as $k => $v) {
        if (!isset($chavesCobertas[$k])) {
            $resto[$k] = $v;
        }
    }
    if (!empty($resto)) {
        print_all_keys_generic($resto, 'Campos adicionais do pedido', $ind1);
    }
}

/* -------------------- Varredura do diretrio -------------------- */

if (!is_dir($dir)) {
    fwrite(STDERR, "[ERRO] Diretrio no encontrado: {$dir}\n");
    exit(1);
}

$dh = opendir($dir);
if ($dh === false) {
    fwrite(STDERR, "[ERRO] No foi possvel abrir o diretrio: {$dir}\n");
    exit(1);
}

while (($entry = readdir($dh)) !== false) {
    // ignora . e ..
    if ($entry === '.' || $entry === '..') continue;

    $path = $dir . DIRECTORY_SEPARATOR . $entry;

    if (!is_file($path)) continue;

    // extenso .json
    $ext = pathinfo($entry, PATHINFO_EXTENSION);
    if (strtolower($ext) !== 'json') continue;

    // comea com prefixo (sem usar str_starts_with)
    if (substr($entry, 0, strlen($prefix)) !== $prefix) continue;

    // confere padro completo
    if (!preg_match($regex, $entry)) continue;

    echo PHP_EOL . "=========================" . PHP_EOL;
    echo "Arquivo: {$entry}" . PHP_EOL;
    echo "Caminho : {$path}" . PHP_EOL;

    try {
        $raw  = file_get_contents($path);
        if ($raw === false) {
            throw new RuntimeException("Falha ao ler o arquivo.");
        }

        $json = decode_json_strict($raw);

        // Estrutura esperada: {"pedidos":[ {...}, {...} ]}
        if (empty($json['pedidos']) || !is_array($json['pedidos'])) {
            echo "[AVISO] Campo 'pedidos' ausente ou no  array." . PHP_EOL;
            continue;
        }

        foreach ($json['pedidos'] as $i => $pedido) {
            echo PHP_EOL . "---------- Pedido #" . ($i+1) . " ----------" . PHP_EOL;
            if (!is_array($pedido)) {
                print_kv('pedido', $pedido);
                continue;
            }
            print_pedido($pedido, 0);
        }

    } catch (Throwable $e) {
        echo "[ERRO] {$entry}: {$e->getMessage()}" . PHP_EOL;
        // opcional: log detalhado
        // file_put_contents(__DIR__.'/erro_json.log', $entry.' => '.$e->getMessage().PHP_EOL, FILE_APPEND);
    }
}

closedir($dh);

echo PHP_EOL . "Concludo." . PHP_EOL;
