<?php
require_once("RetornoBase.php");

/**Classe abstrata para leitura de arquivos de retorno de cobranças no padrão CNAB400/CBR643.<br/>
* Layout Padrão CNAB/Febraban 400 posições<br/>.
* Baseado na documentação para "Layout de Arquivo Retorno para Convênios
* na faixa numérica entre 000.001 a 999.999 (Convênios de até 6 posições). Versão Set/09" e
* "Layout de Arquivo Retorno para convênios na faixa numérica entre 1.000.000 a 9.999.999
* (Convênios de 7 posições). Versão Set/09" do Banco do Brasil 
* (arquivos Doc8826BR643Pos6.pdf e Doc2628CBR643Pos7.pdf)
* @copyright GPLv2
* @package ArquivoRetornoTitulosBancarios
* @author Manoel Campos da Silva Filho. http://manoelcampos.com/contato
* @version 0.4
*/
abstract class RetornoCNAB400Base extends RetornoBase {
   /**@property int HEADER_ARQUIVO Define o valor que identifica uma coluna do tipo HEADER DE ARQUIVO*/
   const HEADER_ARQUIVO = 0;
   /**@property int TRAILER_ARQUIVO Define o valor que identifica uma coluna do tipo TRAILER DE ARQUIVO*/
   const TRAILER_ARQUIVO = 9;

   public function __construct($nomeArquivo=NULL, $aoProcessarLinhaFunctionName=""){
      parent::__construct($nomeArquivo, $aoProcessarLinhaFunctionName);
   }
   /**Processa a linha header do arquivo
   * @param string $sLin Linha do header de arquivo processado
   * @return array<mixed> Retorna um vetor contendo os dados dos campos do header do arquivo.*/
   protected function processarHeaderArquivo($sLin) {
      $aHeader = array();	
      //X = ALFANUMÉRICO 9 = NUMÉRICO V = VÍRGULA DECIMAL ASSUMIDA
      $aHeader["registro"]           = substr($sLin,                    0,     1); //9 Identificação do Registro Header: “0”
      $aHeader["tipo_operacao"]      = substr($sLin,                    1,     1); //9 Tipo de Operação: “2”
      $aHeader["id_tipo_operacao"]   = substr($sLin,                    2,     7); //X Identificação Tipo de Operação “RETORNO”
      $aHeader["id_tipo_servico"]    = substr($sLin,                    9,    2); //9 Identificação do Tipo de Serviço: “01”
      $aHeader["tipo_servico"]       = substr($sLin,                    11,    8); //X Identificação por Extenso do Tipo de Serviço: “COBRANCA”
      $aHeader["complemento1"]       = substr($sLin,                    19,    7); //X Complemento do Registro: “Brancos”
      $aHeader["agencia_cedente"]    = substr($sLin,                    26,    4); //9 Prefixo da Agência: N. Agência onde está cadastrado o convênio líder do cedente
      $aHeader["dv_agencia_cedente"] = substr($sLin,                    30,    1); //X Dígito Verificador - D.V. - do Prefixo da Agência
      $aHeader["conta_cedente"]      = substr($sLin,                    31,    8); //9 Número da Conta Corrente onde está cadastrado o Convênio Líder do Cedente
      $aHeader["dv_conta _cedente"]  = substr($sLin,                    39,    1); //X Dígito Verificador - D.V. - da Conta Corrente do Cedente
      $aHeader["nome_cedente"]       = substr($sLin,                    46,   30); //X Nome do Cedente
      $aHeader["banco"]              = substr($sLin,                    76,   18); //X 001BANCODOBRASIL
      $aHeader["data_gravacao"]      = $this->formataData(substr($sLin, 94,    6)); //9 Data da Gravação: Informe no formado “DDMMAA”
      $aHeader["sequencial_reg"]     = substr($sLin,                   394,    6); //9 Seqüencial do Registro: ”000001”
      return $aHeader;
   }

   /**Processa uma linha detalhe do arquivo.
   * @param string $sLin Linha detalhe do arquivo processado   
   * @return array<mixed> Retorna um vetor contendo os dados dos campos da linha detalhe.*/
   protected function processarDetalhe($sLin) {
      $aDetalhe = array();
      //X = ALFANUMÉRICO 9 = NUMÉRICO V = VÍRGULA DECIMAL ASSUMIDA
      $aDetalhe["registro"]            = substr($sLin,                        0,    1); //9  Id do Registro Detalhe: 1 p/ convênios de 6 dígitos e 7 para convênios de 7 dígitos
      //$aDetalhe["zeros1"]              = substr($sLin,                        1,    2); //9  Zeros
      //$aDetalhe["zeros2"]              = substr($sLin,                        3,   14); //9  Zeros
      $aDetalhe["agencia"]             = substr($sLin,                       17,    4); //9  Prefixo da Agência
      $aDetalhe["dv_agencia"]          = substr($sLin,                       21,    1); //X  Dígito Verificador - D.V. - do Prefixo da Agência
      $aDetalhe["cc_cedente"]          = substr($sLin,                       22,    8); //9  Número da Conta Corrente do Cedente
      $aDetalhe["dv_cc_cedente"]       = substr($sLin,                       30,    1); //X  Dígito Verificador - D.V. - do Número da Conta Corrente do Cedente
      $aDetalhe["taxa_desconto"]       = $this->formataNumero(substr($sLin,  95,    5)); //9  v99 Taxa de desconto
      $aDetalhe["taxa_iof"]            = substr($sLin,                      100,    5); //9  Taxa de IOF
      //$aDetalhe["branco"]              = substr($sLin,                      106,    1); //x  Branco
      $aDetalhe["carteira"]            = substr($sLin,                      106,    2); //9  Carteira
      $aDetalhe["comando"]             = substr($sLin,                      108,    2); //9  Comando - nota 07
      $aDetalhe["data_ocorrencia"]     = $this->formataData(substr($sLin,   110,    6)); //X  Data da Entrada/Liquidação (DDMMAA)
      $aDetalhe["num_titulo"]          = substr($sLin,                      116,   10); //X  Número título dado pelo cedente - (ver nota 06 para convênio de 6 dígitos)
      $aDetalhe["data_vencimento"]     = substr($sLin,                      146,    6); //9  Data de vencimento (DDMMAA) (ver nota 6 para convênios de 7 dígitos)
      $aDetalhe["valor"]               = $this->formataNumero(substr($sLin, 152,   13)); //9  v99 Valor do título
      $aDetalhe["cod_banco"]           = substr($sLin,                      165,    3); //9  Código do banco recebedor - ver nota 08
      $aDetalhe["agencia"]             = substr($sLin,                      168,    4); //9  Prefixo da agência recebedora - ver nota 08
      $aDetalhe["dv_agencia"]          = substr($sLin,                      172,    1); //X  DV prefixo recebedora
      $aDetalhe["especie"]             = substr($sLin,                      173,    2); //9  Espécie do título - ver nota 09
      $aDetalhe["data_credito"]        = substr($sLin,                      175,    6); //9  Data do crédito (DDMMAA) - ver nota 10
      $aDetalhe["valor_tarifa"]        = $this->formataNumero(substr($sLin, 181,    7)); //9  v99 Valor da tarifa - ver nota 06
      $aDetalhe["outras_despesas"]     = $this->formataNumero(substr($sLin, 188,   13)); //9  v99 Outras despesas
      $aDetalhe["juros_desconto"]      = $this->formataNumero(substr($sLin, 201,   13)); //9  v99 Juros do desconto
      $aDetalhe["iof_desconto"]        = $this->formataNumero(substr($sLin, 214,   13)); //9  v99 IOF do desconto
      $aDetalhe["valor_abatimento"]    = $this->formataNumero(substr($sLin, 227,   13)); //9  v99 Valor do abatimento
      $aDetalhe["desconto_concedido"]  = $this->formataNumero(substr($sLin, 240,   13)); //9  v99 Desconto concedido 
      $aDetalhe["valor_recebido"]      = $this->formataNumero(substr($sLin, 253,   13)); //9  v99 Valor recebido (valor recebido parcial)
      $aDetalhe["juros_mora"]          = $this->formataNumero(substr($sLin, 266,    13)); //9  v99 Juros de mora
      $aDetalhe["outros_recebimentos"] = $this->formataNumero(substr($sLin, 279,    13)); //9  v99 Outros recebimentos
      $aDetalhe["abatimento_nao_aprov"]= $this->formataNumero(substr($sLin, 292,    13)); //9  v99 Abatimento não aproveitado pelo sacado
      $aDetalhe["valor_lancamento"]    = $this->formataNumero(substr($sLin, 305,    13)); //9  v99 Valor do lançamento
      $aDetalhe["indicativo_dc"]       = substr($sLin,                      318,    1); //9  Indicativo de débito/crédito - ver nota 11
      $aDetalhe["indicador_valor"]     = substr($sLin,                      319,    1); //9  Indicador de valor -ver  nota 12
      $aDetalhe["valor_ajuste"]        = $this->formataNumero(substr($sLin, 320,    12)); //9  v99 Valor do ajuste - ver nota 13
      //$aDetalhe["brancos1"]            = substr($sLin,                      332,    1); //X  Brancos (vide observação para cobrança compartilhada) 14
      //$aDetalhe["brancos2"]            = substr($sLin,                      333,    9); //9  Brancos (vide observação para cobrança compartilhada) 14
      $aDetalhe["canal_pag_titulo"]    = substr($sLin,                      392,    2); //9 Canal de pagamento do título utilizado pelo sacado - ver nota 15
      $aDetalhe["sequencial"]          = substr($sLin,                      394,    6); //9 Seqüencial do registro
      return $aDetalhe;
   }
   /**Processa a linha trailer do arquivo.
   * @param string $sLin Linha trailer do arquivo processado
   * @return array<mixed> Retorna um vetor contendo os dados dos campos da linha trailer do arquivo.*/
   protected function processarTrailerArquivo($sLin) {
      $aTrailer = array();
      //X = ALFANUMÉRICO 9 = NUMÉRICO V = VÍRGULA DECIMAL ASSUMIDA
      $aTrailer["registro"]                = substr($sLin,                        0,   1);  //9  Identificação do Registro Trailer: “9”
      $aTrailer["retorno"]                 = substr($sLin,                        1,   1);  //9  “2”
      $aTrailer["tipo_registro"]           = substr($sLin,                        2,   2);  //9  “01”
      $aTrailer["cod_banco"]               = substr($sLin,                        4,   3);  
      $aTrailer["cob_simples_qtd_titulos"] = substr($sLin,                       17,   8);  //9  Cobrança Simples - quantidade de títulos em cobranca
      $aTrailer["cob_simples_vlr_total"]   = $this->formataNumero(substr($sLin,  25,  14)); //9  v99 Cobrança Simples - valor total
      $aTrailer["cob_simples_num_aviso"]   = substr($sLin,                       39,   8);  //9  Cobrança Simples - Número do aviso
      $aTrailer["cob_vinc_qtd_titulos"]    = substr($sLin,                       57,   8);  //9  Cobrança Vinculada - quantidade de títulos
      $aTrailer["cob_vinc_valor_total"]    = $this->formataNumero(substr($sLin,  65,  14)); //9  v99 Cobrança Vinculada - valor total
      $aTrailer["cob_vinc_num_aviso"]      = substr($sLin,                       79,   8);  //9  Cobrança Vinculada - Número do aviso
      //$aTrailer["cob_vinc_brancos"]        = substr($sLin,                       87,  10); //X  Cobrança Vinculada - Brancos
      $aTrailer["cob_cauc_qtd_titulos"]    = substr($sLin,                       97,   8);  //9  Cobrança Caucionada - quantidade de títulos
      $aTrailer["cob_cauc_vlr_total"]      = $this->formataNumero(substr($sLin, 105,  14)); //9  v99 Cobrança Caucionada - valor total
      $aTrailer["cob_cauc_num_aviso"]      = substr($sLin,                      119,   8);  //9  Cobrança Caucionada - Número do aviso
      //$aTrailer["cob_cauc_brancos"]        = substr($sLin,                      127,  10); //X  Cobrança Caucionada - Brancos
      $aTrailer["cob_desc_qtd_titulos"]    = substr($sLin,                      137,   8);  //9  Cobrança Descontada - quantidade de títulos
      $aTrailer["cob_desc_vlr_total"]      = $this->formataNumero(substr($sLin, 145,  14)); //9  v99 Cobrança Descontada - valor total
      $aTrailer["cob_desc_num_aviso"]      = substr($sLin,                      159,   8);  //9  Cobrança Descontada - Número do aviso
      //$aTrailer["cob_desc_brancos"]        = substr($sLin,                      167,  50); //X  Cobrança Descontada - Brancos
      $aTrailer["cob_vendor_qtd_titulos"]  = substr($sLin,                      217,   8);  //9  Cobrança Vendor - quantidade de títulos
      $aTrailer["cob_vendor_vlr_total"]    = $this->formataNumero(substr($sLin, 225,  14)); //9  v99 Cobrança Vendor - valor total
      $aTrailer["cob_vendor_num_aviso"]    = substr($sLin,                      239,   8);  //9  Cobrança Vendor - Número do aviso
      //$aTrailer["cob_vendor_brancos"]      = substr($sLin,                      247, 147); //X  Cobrança Vendor – Brancos
      $aTrailer["sequencial"]              = substr($sLin,                      394,   6);  //9  Seqüencial do registro
      return $aTrailer;
   }

   /**Processa uma linha do arquivo de retorno.
   * @param int $numLn Número_linha a ser processada
   * @param string $sLin String contendo a linha a ser processada
   * @return array Retorna um vetor associativo contendo os valores_linha processada.*/
   public function processarLinha($iNumLin, $sLin) {
      $iTamLinha = 400; //total de caracteres das linhas do arquivo
      //o +2 é utilizado para contar o \r\n no final da linha
      if(strlen($sLin)!=$iTamLinha and strlen($sLin)!=$iTamLinha+2) die("A linha $iNumLin no tem $iTamLinha posies. Possui " . strlen($sLin));
      if(trim($sLin)=="")die("A linha $iNumLin est vazia.");
      //é adicionado um espaço vazio no início_linha para que
      //possamos trabalhar com índices iniciando_1, no lugar_zero,
      //e assim, ter os valores_posição_campos exatamente
      //como no manual CNAB400
      $sLin = " $sLin";
      $sTipoLin = substr($sLin,  1,  1);
      if($sTipoLin == RetornoCNAB400::HEADER_ARQUIVO) {
         $sRet = $this->processarHeaderArquivo($sLin);
      } else if($sTipoLin == RetornoCNAB400::DETALHE) {
         $sRet = $this->processarDetalhe($sLin);
      } else if($sTipoLin == RetornoCNAB400::TRAILER_ARQUIVO) {
         $sRet = $this->processarTrailerArquivo($sLin); 
      } else {
         $sRet = NULL;
      }
      return $sRet;
   }
}

?>
